/**
* \file ProfisharkAPI.h
* Main API header file
*/
#pragma once

#if defined(_WIN32) || defined(WIN32)
	#ifdef PROFISHARKAPI_EXPORTS
		#define PROFISHARKAPI_API __declspec(dllexport)
	#else
		#define PROFISHARKAPI_API __declspec(dllimport)
	#endif
#endif
#ifdef __unix__  
	#ifdef PROFISHARKAPI_EXPORTS
		#define PROFISHARKAPI_API __attribute__((visibility("default")))
	#else
		#define PROFISHARKAPI_API 
	#endif
#endif

#ifdef __APPLE__
    #define PROFISHARKAPI_API __attribute__((visibility("default")))
#endif

#include <stdint.h>

extern "C" {
/**
* \brief Error constants
*/
enum ProfisharkAPIRet
{
	ProfisharkAPIRet_Success=0,			/**< No error*/
	ProfisharkAPIRet_Failed=-1,			/**< Generic failure*/
	ProfisharkAPIRet_Exist=-2,			/**< Already exists*/
	ProfisharkAPIRet_NotExist=-3,		/**< Does not exist*/
	ProfisharkAPIRet_NoDev=-4,			/**< Device does not exist*/
	ProfisharkAPIRet_Unsupported=-5,	/**< Device does not support this operation */
	ProfisharkAPIRet_CannotCreateFile=-6,		/**< Failed to create file */
	ProfisharkAPIRet_CannotAllocateCache=-7,	/**< Failed to allocate cache, out of memory */
	ProfisharkAPIRet_CaptureNotReady=-8,		/**< Capture is not ready to start */
	ProfisharkAPIRet_CacheTooSmall=-9,			/**< Cache is too small (<512kb) */
	ProfisharkAPIRet_TooSmall=-10,			/**< Not enough space */
	ProfisharkAPIRet_NotImplemented=-11, 	/**< Not implemented in API */
	ProfisharkAPIRet_NotReady=-12, 			/**< Buffer not ready */
	ProfisharkAPIRet_Again=-13, 		/**< Try again */
	ProfisharkAPIRet_CannotOpenFile=-14,		/**< Failed to open file */
	ProfisharkAPIRet_InvalidFirmware=-15,		/**< Invalid firmware file */
	ProfisharkAPIRet_NotStarted=-16,		/**< Invalid firmware file */
};

/**
* \brief State of a flag
*/
enum ProfisharkAPIFlagState
{
	ProfisharkAPIFlagState_No,		/**< Feature is OFF/not OK/not detected */
	ProfisharkAPIFlagState_Yes,		/**< Feature is ON/OK/detected */
	ProfisharkAPIFlagState_Unknown,		/**< State is not available/unknown */
};

/**
* \ingroup info
* \brief Profishark devices models
*/
enum ProfisharkAPIDeviceType
{
	ProfisharkAPIDeviceType_None,							/**< No Profishark device */
	ProfisharkAPIDeviceType_BadDevice,						/**< Profishark device with bad hardware firmware */
	ProfisharkAPIDeviceType_UnknownDevice,					/**< Unknown Profishark device */
	ProfisharkAPIDeviceType_Profishark1G,					/**< Profishark 1G */
	ProfisharkAPIDeviceType_C1GIGE_USB,						/**< C1GIGE-USB */
	ProfisharkAPIDeviceType_Profishark1GPlus,				/**< Profishark-1G+ */
	ProfisharkAPIDeviceType_Profishark1G_v5,				/**< Profishark-1G v5 */
	ProfisharkAPIDeviceType_Profishark1G_v5_POE,			/**< Profishark-1G v5 with PoE board */
	ProfisharkAPIDeviceType_Profishark1GPlus_POE,			/**< Profishark-1G+ with PoE board */
	ProfisharkAPIDeviceType_Profishark1GPlus_GPS,			/**< Profishark-1G+ with timing board */
	ProfisharkAPIDeviceType_Profishark1GPlus_GPS_POE,		/**< Profishark-1G+ with PoE and timing boards */
	ProfisharkAPIDeviceType_Profishark10G,					/**< Profishark-10G */
	ProfisharkAPIDeviceType_Profishark10G_GPS,				/**< Profishark-10G with timing board */
	ProfisharkAPIDeviceType_Profishark100,					/**< Profishark-100 */
	ProfisharkAPIDeviceType_Profishark1G_v6,				/**< Profishark-1G v6 */
};

/**
* \ingroup info
* \brief Profishark device capabilities
*/
enum ProfisharkApiDeviceOptions
{
	ProfisharkApiDeviceOptions_Basic=0x1,						/**< Basic features, covers 'Features' and 'Capture' tabs in Profishark Manager */
	ProfisharkApiDeviceOptions_DualPorts=0x2,					/**< The device has two network ports */
	ProfisharkApiDeviceOptions_SpanMode=0x4,					/**< The device supports span mode */
	ProfisharkApiDeviceOptions_Gps=0x8,							/**< The device has a timing board */
	ProfisharkApiDeviceOptions_Update=0x10,						/**< The device support firmware update (only using Profishark Manager) */
	ProfisharkApiDeviceOptions_Poe=0x20,						/**< The device has a PoE board */
	ProfisharkApiDeviceOptions_Sfp=0x40,						/**< The device has SFP ports */
	ProfisharkApiDeviceOptions_Filter=0x80,						/**< The device supports filtering */
	ProfisharkApiDeviceOptions_StatisticsV1=0x100,				/**< The device supports V1 statistics */
	ProfisharkApiDeviceOptions_StatisticsV2=0x200,				/**< The device supports V2 statistics */
	ProfisharkApiDeviceOptions_SpeedControl=0x400,				/**< The device can control network ports speed */
	ProfisharkApiDeviceOptions_Loopback=0x800,					/**< The device can set notwork ports in loopback mode */
	ProfisharkApiDeviceOptions_Phy88E3015=0x2000,				/**< The device uses a 88E3015 PHY */
	ProfisharkApiDeviceOptions_PhyVSC8641=0x4000,				/**< The device uses a VSC8641 PHY */
	ProfisharkApiDeviceOptions_PcapNG=0x8000,					/**< The device can output pcap-ng data */
	ProfisharkApiDeviceOptions_SlicingLength=0x10000,			/**< The device can change packet slicing length */
	ProfisharkApiDeviceOptions_PhyVSC8502=0x20000,				/**< The device uses a VSC8641 PHY */
	ProfisharkApiDeviceOptions_MultipleHW=0x40000,				/**< The device supports multiple HW firmwares */
};

/**
* \ingroup info
* \brief Profishark device extra information
*/
struct ProfisharkAPIDeviceExtraInfo
{
	uint8_t usb_mode;		/**< USB mode. 1 : Full speed, 2 : High speed, 3 : Super speed */
};

/**
* \ingroup timing
* \brief Time source for the timestamps
*/
enum ProfisharkAPITimeSource
{
	ProfisharkAPITimeSource_System,					/**< System clock */
	ProfisharkAPITimeSource_RTC,					/**< Internal RTC */
	ProfisharkAPITimeSource_GPS,					/**< GPS time */
	ProfisharkAPITimeSource_None,					/**< Not initialized */
	ProfisharkAPITimeSource_Manual,					/**< Manual timestamp */
	ProfisharkAPITimeSource_SNTP,					/**< From SNTP */
};

/**
* \brief Enabled/Disabled states
*/
enum ProfisharkAPIState
{
	ProfisharkAPIState_Disabled=0,		/**< Disabled */
	ProfisharkAPIState_Enabled=1,		/**< Enabled */
	ProfisharkAPIState_DontCare=-1,		/**< Don't Care */
};

/**
* \ingroup features
* \brief Defines when the timestamp information is added to the packet
*/
enum ProfisharkAPITimestampMoment
{
	ProfisharkAPITimestampMoment_Capture,	/**< Timestamp is applied at capture (default) */
	ProfisharkAPITimestampMoment_Ingress,	/**< Timestamp is applied on packet ingress */
	ProfisharkAPITimestampMoment_Egress,	/**< Timestamp is applied on packet egress */
};

/**
* \ingroup features
* \brief Profishark device features
*/
struct ProfisharkAPIDeviceFeatures
{
	ProfisharkAPIState TransmitCrcErrors;		/**< When Enabled, packets with CRC errors are captured, otherwise they are dropped */
	ProfisharkAPIState KeepCRC32Field;			/**< When Enabled, the CRC32 field of ethernet packets is preserved */
	ProfisharkAPIState DisablePortA;			/**< When Enabled, all traffic from port A is dropped */
	ProfisharkAPIState DisablePortB;			/**< When Enabled, all traffic from port B is dropped */
	ProfisharkAPIState Slicing;					/**< When Enabled, all captured packets are truncated */
	ProfisharkAPIState InlineMode;				/**< When Disabled, ports act in span mode (only available if SpanMode is defined in #ProfisharkApiDeviceOptions) */
	ProfisharkAPIState Loopback;				/**< When Enabled, ports are in loopback mode (only available if Loopback is defined in #ProfisharkApiDeviceOptions) */
	ProfisharkAPIState EnableTimestamps;		/**< When Enabled, add timestamps to captured packets */

	ProfisharkAPIState ForcePPSGeneration;		/**< When Enabled, force PPS generation even if GPS signal is not locked (only available if Gps is defined in #ProfisharkApiDeviceOptions)*/
	ProfisharkAPITimeSource TimeInitSource;		/**< Select source for the time information at startup (only available if Gps is defined in #ProfisharkApiDeviceOptions). Only ProfisharkAPITimeSource_System, ProfisharkAPITimeSource_RTC, ProfisharkAPITimeSource_GPS and ProfisharkAPITimeSource_None are accepted */
	ProfisharkAPIState WaitForSync;				/**< When Enabled, wait for the timestamping mechanism to be synchronized with PPS time before starting (only available if Gps is defined in #ProfisharkApiDeviceOptions) */
	ProfisharkAPIState PPSOutput;				/**< When Enabled, the PPS port acts as output */
	uint16_t PPSCompensation;					/**< PPS cable length compensation. This is the number of nanoseconds with a resolution of 2^-32 (i.e. 0x0001 means -0.23283064365386962890625ns) (only available if Gps is defined in #ProfisharkApiDeviceOptions) */
	ProfisharkAPITimestampMoment PortATimestampMoment;	/**< Defines when the timestamp is applied to packets from port A (only available if Gps is defined in #ProfisharkApiDeviceOptions) */
	ProfisharkAPITimestampMoment PortBTimestampMoment;	/**< Defines when the timestamp is applied to packets from port B (only available if Gps is defined in #ProfisharkApiDeviceOptions) */
	ProfisharkAPIState StatisticsOnly;			/**< The Profishark driver does not transfer data to ndis. It only generates statistics. */

	uint16_t SlicingLength;						/**< Length of the sliced packets. Available only on Profishark 10G and when Slicing is enabled */

	uint8_t HWFirmwareId;						/**< Id of the selected HW firmware. Changes to this value are only applied on reset. Must be in [0..3]*/
};

/**
* \ingroup features
* \brief Profishark port speed
*/
struct ProfisharkAPIDevicePortSpeed
{
	ProfisharkAPIState Autonegotiation;		/**< Set autonegotiation status*/
	ProfisharkAPIState _1000_FullDuplex;	/**< Set Gigabit ethernet status*/
	ProfisharkAPIState _100_FullDuplex;		/**< Set Fast ethernet full duplex status*/
	ProfisharkAPIState _100_HalfDuplex;		/**< Set Fast ethernet half duplex status*/
	ProfisharkAPIState _10_FullDuplex;		/**< Set ethernet full duplex status*/
	ProfisharkAPIState _10_HalfDuplex;		/**< Set ethernet half duplex status*/
	ProfisharkAPIState AsymmetricPause;		/**< Set asymmetric pause status*/
	ProfisharkAPIState SymmetricPause;		/**< Set symmetric pause status*/
	ProfisharkAPIState PreferMaster;		/**< When set, the port PHY will prefer to be master of the link*/
	ProfisharkAPIState ForceMasterSlave;	/**< When set, the port PHY will force the state of PreferState*/
};

/**
* \ingroup features
* \brief Profishark ports speed
*/
struct ProfisharkAPIDevicePortsSpeed
{
	ProfisharkAPIDevicePortSpeed PortsSpeed[2];
};

/**
* \ingroup info
* \brief SW firmware or driver version
*/
struct ProfisharkApiSWVersion
{
	uint8_t v[4];	/**< Version, v[0] is the MSB, v[4] is the LSB */
};

/**
* \ingroup info
* \brief HW firmware version
*/
struct ProfisharkApiHWVersion
{
	uint32_t v;
};

/**
* \ingroup info
* \brief firmwares and driver versions
*/
struct ProfisharkApiVersions
{
	ProfisharkApiSWVersion swFirmware;	/**< SW firmware version */
	ProfisharkApiSWVersion driver;		/**< Driver version */
	ProfisharkApiHWVersion hwFirmware;	/**< HW firmware version */
};

/**
* \ingroup info
* \brief Port Mode
*/
enum ProfisharkApiPortMode
{
	ProfisharkApiPortMode_Inline,	/**< Inline */
	ProfisharkApiPortMode_Span,		/**< Span */
};

/**
* \ingroup info
* \brief Link speed
*/
enum ProfisharkApiPortSpeed
{
	ProfisharkApiPortSpeed_10Mbps,		/**< 10Mbps */
	ProfisharkApiPortSpeed_100Mbps,		/**< 100Mbps */
	ProfisharkApiPortSpeed_1Gbps,		/**< 1Gbps */
	ProfisharkApiPortSpeed_10Gbps,		/**< 10Gbps */
	ProfisharkApiPortSpeed_Unknown,		/**< Unknown link speed */
	ProfisharkApiPortSpeed_Down,		/**< Link is down */
};

/**
* \ingroup info
* \brief Link duplex
*/
enum ProfisharkApiPortDuplex
{
	ProfisharkApiPortDuplex_Full,	/**< Full duplex */
	ProfisharkApiPortDuplex_Half,	/**< Half duplex */
};

/**
* \ingroup info
* \brief RJ45 port link status
*/
struct ProfisharkApiRJ45PortStatus
{
	ProfisharkApiPortMode mode;			/**< Inline or Span */
	ProfisharkApiPortSpeed speed;		/**< Link speed */
	ProfisharkApiPortDuplex duplex;		/**< Link duplex */
};

/**
* \ingroup info
* \brief Describe all RJ45 ports
*/
struct ProfisharkApiRJ45PortsStatus
{
	bool valid;										/**< The rest of the structure is only valid if 'valid' is true. False means RJ45 ports are not available */
	int portsCount;									/**< Number of available RJ45 ports */
	ProfisharkApiRJ45PortStatus portsStatus[2];		/**< Array containing the ports link status */
};

/**
* \ingroup info
* \brief SFP module encoding.
*
* Values over 0x06 are Reserved
*/
enum ProfisharkApiSFPPortsStatusEncoding
{
	ProfisharkApiSFPPortsStatusEncoding_Unspecified=0x00,
	ProfisharkApiSFPPortsStatusEncoding_8B10B=0x01,
	ProfisharkApiSFPPortsStatusEncoding_4B5B=0x02,
	ProfisharkApiSFPPortsStatusEncoding_NRZ=0x03,
	ProfisharkApiSFPPortsStatusEncoding_Manchester=0x04,
	ProfisharkApiSFPPortsStatusEncoding_SONET_Scrambled=0x05,
	ProfisharkApiSFPPortsStatusEncoding_64B66B=0x06,
};

/**
* \ingroup info
* \brief SFP module rate ID.
*
* Other values are Unallocated
*/
enum ProfisharkApiSFPPortsRateID
{
	ProfisharkApiSFPPortsRateID_Unspecified=0x00,
	ProfisharkApiSFPPortsRateID_SFF_8079=0x01,
	ProfisharkApiSFPPortsRateID_SFF_8431_RXOnly=0x02,
	ProfisharkApiSFPPortsRateID_SFF_8431_TXOnly=0x04,
	ProfisharkApiSFPPortsRateID_SFF_8431=0x06,
	ProfisharkApiSFPPortsRateID_FC_PI_5_RXOnly=0x08,
	ProfisharkApiSFPPortsRateID_FC_PI_5=0x0a,
};

/**
* \ingroup info
* \brief SFP module identifier.
*
* Other values under 0x80 are Reserved
* Values from 0x80 are Vendor specific
*/
enum ProfisharkApiSFPPortsIdentifier
{
	ProfisharkApiSFPPortsIdentifier_Unknown,
	ProfisharkApiSFPPortsIdentifier_GBIC,
	ProfisharkApiSFPPortsIdentifier_SolderedToMB,
	ProfisharkApiSFPPortsIdentifier_SFP_SFPPlus,
	ProfisharkApiSFPPortsIdentifier_300pinXBI,
	ProfisharkApiSFPPortsIdentifier_XENPAK,
	ProfisharkApiSFPPortsIdentifier_XFP,
	ProfisharkApiSFPPortsIdentifier_XFF,
	ProfisharkApiSFPPortsIdentifier_XFP_E,
	ProfisharkApiSFPPortsIdentifier_XPAK,
	ProfisharkApiSFPPortsIdentifier_X2,
	ProfisharkApiSFPPortsIdentifier_DWDM_SFP,
	ProfisharkApiSFPPortsIdentifier_QSFP,
	ProfisharkApiSFPPortsIdentifier_QSFPPlus,
	ProfisharkApiSFPPortsIdentifier_CXP,
};

/**
* \ingroup info
* \brief SFP module connector.
*
* Other values under 0x80 are Reserved
* Values from 0x80 are Vendor specific
*/
enum ProfisharkApiSFPPortsConnector
{
	ProfisharkApiSFPPortsConnector_Unknown,
	ProfisharkApiSFPPortsConnector_SC,
	ProfisharkApiSFPPortsConnector_FCStyle1Copper,
	ProfisharkApiSFPPortsConnector_FCStyle2Copper,
	ProfisharkApiSFPPortsConnector_BNC_TNC,
	ProfisharkApiSFPPortsConnector_FCCoax,
	ProfisharkApiSFPPortsConnector_FiberJack,
	ProfisharkApiSFPPortsConnector_LC,
	ProfisharkApiSFPPortsConnector_MT_RJ,
	ProfisharkApiSFPPortsConnector_MU,
	ProfisharkApiSFPPortsConnector_SG,
	ProfisharkApiSFPPortsConnector_OpticalPigtail,
	ProfisharkApiSFPPortsConnector_MPOParallelOptic,
	ProfisharkApiSFPPortsConnector_HSSDC_II,
	ProfisharkApiSFPPortsConnector_CopperPigtail,
	ProfisharkApiSFPPortsConnector_RJ45,
};

/**
* \ingroup info
* \brief Diagnostic monitoring type
*/
enum ProfisharkApiSFPPortsDiagMonitoringType
{
	ProfisharkApiSFPPortsDiagMonitoringType_IntCalibrated_AVPower,	/**< Internally calibrated, Average power */
	ProfisharkApiSFPPortsDiagMonitoringType_IntCalibrated_OMA,	/**< Internally calibrated, OMA */
	ProfisharkApiSFPPortsDiagMonitoringType_ExtCalibrated_AVPower,	/**< Externally calibrated, Average power */
	ProfisharkApiSFPPortsDiagMonitoringType_ExtCalibrated_OMA,	/**< Externally calibrated, OMA */
	ProfisharkApiSFPPortsDiagMonitoringType_None,				/**< No monitoring */
};

/**
* \ingroup info
* \brief SFF8472  compliance
*
* Other values are unknown.
*/
enum ProfisharkApiSFPPortsSFF8472
{
	ProfisharkApiSFPPortsSFF8472_Undefined,	
	ProfisharkApiSFPPortsSFF8472_Rev9_3,	
	ProfisharkApiSFPPortsSFF8472_Rev9_5,
	ProfisharkApiSFPPortsSFF8472_Rev10_2,
	ProfisharkApiSFPPortsSFF8472_Rev10_4,	
	ProfisharkApiSFPPortsSFF8472_Rev_11_0,	
};

/**
* \ingroup info
* \brief SFP module alarm, warning and gpios
*/
enum ProfisharkApiSFPPortsDiagFlags
{
	ProfisharkApiSFPPortsDiagFlags_Alarm_Temperature=0x1,
	ProfisharkApiSFPPortsDiagFlags_Alarm_VCC=0x2,
	ProfisharkApiSFPPortsDiagFlags_Alarm_TXBias=0x4,
	ProfisharkApiSFPPortsDiagFlags_Alarm_TXPower=0x8,
	ProfisharkApiSFPPortsDiagFlags_Alarm_RXPower=0x10,
	ProfisharkApiSFPPortsDiagFlags_Warning_Temperature=0x20,
	ProfisharkApiSFPPortsDiagFlags_Warning_VCC=0x40,
	ProfisharkApiSFPPortsDiagFlags_Warning_TXBias=0x80,
	ProfisharkApiSFPPortsDiagFlags_Warning_TXPower=0x100,
	ProfisharkApiSFPPortsDiagFlags_Warning_RXPower=0x200,
	ProfisharkApiSFPPortsDiagFlags_Gpio_TXDisable=0x400,
	ProfisharkApiSFPPortsDiagFlags_Gpio_TXFault=0x800,
	ProfisharkApiSFPPortsDiagFlags_Gpio_RXLOS=0x1000,
};

/**
* \ingroup info
* \brief SFP module Laser type
*/
enum ProfisharkApiSFPPortsOptionsLaserType
{
	ProfisharkApiSFPPortsOptionsLaser_Cooled,	/**< Cooled laser */
	ProfisharkApiSFPPortsOptionsLaser_Uncooled_Unspecified, /**< uncooled laser or unspecified */
};

/**
* \ingroup info
* \brief SFP module Laser power
*/
enum ProfisharkApiSFPPortsOptionsLaserPower
{
	ProfisharkApiSFPPortsOptionsLaserPower_Level2,	/**< Laser power level 2 */
	ProfisharkApiSFPPortsOptionsLaserPower_Level1_Unspecified, /**< Laser power level 1 or unspecified */
};

/**
* \ingroup info
* \brief SFP module receiver output type
*/
enum ProfisharkApiSFPPortsOptionsReceiverOutput
{
	ProfisharkApiSFPPortsOptionsLinearReceiverOutput_Linear,	/**< Receiver has linear output */
	ProfisharkApiSFPPortsOptionsLinearReceiverOutput_Limiting,	/**< Receiver has limiting output */
};

/**
* \ingroup info
* \brief implementation state for various SFP module features
*/
enum ProfisharkApiSFPPortsImplementation
{
	ProfisharkApiSFPPortsImplementation_Implemented,			/**< Feature is implemented */
	ProfisharkApiSFPPortsImplementation_NotImplemented,			/**< Feature is not implemented */
	ProfisharkApiSFPPortsImplementation_Implemented_Inverted,	/**< Feature is implemented, but inverted */
};

/**
* \ingroup info
* \brief SFP module SONET reach specifier
*/
enum ProfisharkApiSFPPortsSONETReachSpecifier
{
	ProfisharkApiSFPPortsSONETReachSpecifier_SR,
	ProfisharkApiSFPPortsSONETReachSpecifier_SR_1,
	ProfisharkApiSFPPortsSONETReachSpecifier_IR_1,
	ProfisharkApiSFPPortsSONETReachSpecifier_IR_2,
	ProfisharkApiSFPPortsSONETReachSpecifier_LR_1,
	ProfisharkApiSFPPortsSONETReachSpecifier_LR_2,
	ProfisharkApiSFPPortsSONETReachSpecifier_LR_3,
	ProfisharkApiSFPPortsSONETReachSpecifier_Unknown,
};

/**
* \ingroup info
* \brief Description of a SFP module
*/
struct ProfisharkApiSFPPortStatus
{
	bool modulePresent;						/**< SFP module is present. If false, the whole structure is invalid */
	char VendorName[17];					/**< Vendor Name*/
	int VendorOUI;					/**< Vendor OUI*/
	char Model[17];						/**< Model Name*/
	char Revision[5];					/**< Revision*/
	char DateCode[14];					/**< Date Code */
	char SerialNo[17];					/**< Serial Number */
	int NominalRate;						/**< Nominale rate in Mbps. 0 means unspecified. */
	int MaximumBitrate;						/**< Maximum bitrate in % of NominalRate. 0 means unspecified. */
	int MinimumBitrate;						/**< Minimum bitrate in % of NominalRate. 0 means unspecified. */
	ProfisharkApiSFPPortsStatusEncoding Encoding;	/**< Data encoding */
	ProfisharkApiSFPPortsRateID RateID;				/**< Rate ID */
	ProfisharkApiSFPPortsIdentifier identifier;		/**< Module identifier */
	int ExtIdentifier;								/**< Extended identifier */
	ProfisharkApiSFPPortsConnector connector;		/**< Connector */
	int WaveLength;									/**< Wavelength in nm. */
	ProfisharkApiSFPPortsDiagMonitoringType MonitoringType;	/**< Monitoring type */
	ProfisharkApiSFPPortsSFF8472 SFF8472Rev;	/**< SFF8472  compliance*/
	int Length9um;								/**< Fibre 9/125um length in meters. 0 means unspecified, -1 means >25.4km, -2 means >254km */
	int Length50um;								/**< OM2 Fibre 50/125um length in meters. 0 means unspecified, -1 means >2.54km */
	int Length625um;							/**< OM1 Fibre 62.5/125um length in meters. 0 means unspecified, -1 means >2.54km */
	int LengthCopper;							/**< Copper/Active cable length in meters. 0 means unspecified, -1 means >254m */
	int Length50um2;							/**< Fibre 50/125um length in meters. 0 means unspecified, -1 means >2.54km */

	ProfisharkApiSFPPortsOptionsLaserType LaserType;						/**< Laser type */
	ProfisharkApiSFPPortsOptionsLaserPower LaserPower;						/**< Laser power */
	ProfisharkApiSFPPortsOptionsReceiverOutput ReceiverOutput;				/**< receiver output type */
	ProfisharkApiSFPPortsImplementation RateSelectImplementation;			/**< Rate select gpio implementation */
	ProfisharkApiSFPPortsImplementation TXDisableImplementation;			/**< TX Disable gpio implementation */
	ProfisharkApiSFPPortsImplementation TXFaultImplementation;				/**< TX Fault gpio implementation */
	ProfisharkApiSFPPortsImplementation LOSImplementation;					/**< LOS gpio implementation */
	ProfisharkApiSFPPortsImplementation AlarmWarningImplementation;			/**< Alarms and warnings implementation */
	ProfisharkApiSFPPortsImplementation SoftTXDisableControlImplementation;		/**< software TX Disable gpio control implementation */
	ProfisharkApiSFPPortsImplementation SoftTXFaultControlImplementation;			/**< software TX Fault gpio control implementation */
	ProfisharkApiSFPPortsImplementation SoftRXLOSControlImplementation;			/**< software RX LOS gpio control implementation */
	ProfisharkApiSFPPortsImplementation SoftRateSelectControlImplementation;		/**< software Rate select gpio control implementation */
	ProfisharkApiSFPPortsImplementation ApplicationSelectImplementation;	/**< software application selection implementation */
	ProfisharkApiSFPPortsImplementation SoftRateSelectImplementation;		/**< software rate control implementation */

	ProfisharkApiSFPPortsImplementation Ethernet10G_BaseER;									/**< 10G BASE-ER support */
	ProfisharkApiSFPPortsImplementation Ethernet10G_BaseLRM;								/**< 10G BASE-LRM support */
	ProfisharkApiSFPPortsImplementation Ethernet10G_BaseLR;									/**< 10G BASE-LR support */
	ProfisharkApiSFPPortsImplementation Ethernet10G_BaseSR;									/**< 10G BASE-SR support */
	ProfisharkApiSFPPortsImplementation Infiniband_1X_SX;									/**< Infiniband 1X SX  support */
	ProfisharkApiSFPPortsImplementation Infiniband_1X_LX;									/**< Infiniband 1X LX  support */
	ProfisharkApiSFPPortsImplementation Infiniband_1X_CopperActive;							/**< Infiniband Active copper support */
	ProfisharkApiSFPPortsImplementation Infiniband_1X_CopperPassive;						/**< Infiniband Passive copper support */
	ProfisharkApiSFPPortsImplementation SONET_OC192_Short;						/**< SONET OC-192 short reach support */
	ProfisharkApiSFPPortsImplementation SONET_OC48_Long;							/**< SONET OC-48 long reach support */
	ProfisharkApiSFPPortsImplementation SONET_OC48_Intermediate;					/**< SONET OC-48 intermediate reach support */
	ProfisharkApiSFPPortsImplementation SONET_OC48_Short;						/**< SONET OC-48 short reach support */
	ProfisharkApiSFPPortsImplementation SONET_OC12_SingleMode_Long;				/**< SONET OC-12 single mode long reach support */
	ProfisharkApiSFPPortsImplementation SONET_OC12_SingleMode_Intermediate;		/**< SONET OC-12 single mode intermediate reach support */
	ProfisharkApiSFPPortsImplementation SONET_OC12_Short;						/**< SONET OC-12 short reach support */
	ProfisharkApiSFPPortsImplementation SONET_OC3_SingleMode_Long;				/**< SONET OC-3 single mode long reach support */
	ProfisharkApiSFPPortsImplementation SONET_OC3_SingleMode_Intermediate;		/**< SONET OC-3 single mode intermediate reach support */
	ProfisharkApiSFPPortsImplementation SONET_OC3_Short;							/**< SONET OC-3 short reach support */
	ProfisharkApiSFPPortsSONETReachSpecifier SONETReachSpecifier;				/**< SONET reach specifier */
	ProfisharkApiSFPPortsImplementation Ethernet1G_BasePX;									/**< 1G BASE-PX support */
	ProfisharkApiSFPPortsImplementation Ethernet1G_BaseBX10;								/**< 1G BASE-BX10 support */
	ProfisharkApiSFPPortsImplementation Ethernet1G_BaseFX;									/**< 1G BASE-FX support */
	ProfisharkApiSFPPortsImplementation Ethernet1G_BaseLX_LX10;								/**< 1G BASE-LX/LX10 support */
	ProfisharkApiSFPPortsImplementation Ethernet1G_BaseT;									/**< 1G BASE-T support */
	ProfisharkApiSFPPortsImplementation Ethernet1G_BaseCX;									/**< 1G BASE-CX support */
	ProfisharkApiSFPPortsImplementation Ethernet1G_BaseLX;									/**< 1G BASE-LX support */
	ProfisharkApiSFPPortsImplementation Ethernet1G_BaseSX;									/**< 1G BASE-SX support */
	ProfisharkApiSFPPortsImplementation ESCON_MMF;											/**< ESCON MMF 1310nm LED support */
	ProfisharkApiSFPPortsImplementation ESCON_SMF;											/**< ESCON SMF 1310nm Laser support */
	ProfisharkApiSFPPortsImplementation FibreTechnology_SA;									/**< Fibre channel shortware laser, linear RX transmitter support */
	ProfisharkApiSFPPortsImplementation FibreTechnology_LC;									/**< Fibre channel longwave laser (LC) transmitter support */
	ProfisharkApiSFPPortsImplementation FibreTechnology_EL_InterEnclosure;					/**< Fibre channel electrical inter-enclosure transmitter support */
	ProfisharkApiSFPPortsImplementation FibreTechnology_EL_IntraEnclosure;					/**< Fibre channel electrical intra-enclosure transmitter support */
	ProfisharkApiSFPPortsImplementation FibreTechnology_SN;									/**< Fibre channel shortware laser without OFC transmitter support */
	ProfisharkApiSFPPortsImplementation FibreTechnology_SL;									/**< Fibre channel shortware laser with OFC transmitter support */
	ProfisharkApiSFPPortsImplementation FibreTechnology_LL;									/**< Fibre channel longwave laser (LL) transmitter support */
	ProfisharkApiSFPPortsImplementation FibreLength_V;										/**< Fibre channel very long distance support */
	ProfisharkApiSFPPortsImplementation FibreLength_S;										/**< Fibre channel short distance support */
	ProfisharkApiSFPPortsImplementation FibreLength_I;										/**< Fibre channel intermediate distance support */
	ProfisharkApiSFPPortsImplementation FibreLength_L;										/**< Fibre channel long distance support */
	ProfisharkApiSFPPortsImplementation FibreLength_M;										/**< Fibre channel medium distance support */
	ProfisharkApiSFPPortsImplementation SFPPlus_Active;										/**< SFP+ Active cable support */
	ProfisharkApiSFPPortsImplementation SFPPlus_Passive;									/**< SFP+ Passive cable support */
	ProfisharkApiSFPPortsImplementation FibreMedia_TW;										/**< Fibre channel twin axial pair media support */
	ProfisharkApiSFPPortsImplementation FibreMedia_TP;										/**< Fibre channel shielded twisted pair media support */
	ProfisharkApiSFPPortsImplementation FibreMedia_MI;										/**< Fibre channel miniature coax media support */
	ProfisharkApiSFPPortsImplementation FibreMedia_TV;										/**< Fibre channel video coax media support */
	ProfisharkApiSFPPortsImplementation FibreMedia_M6;										/**< Fibre channel multi-mode 62.5um media support */
	ProfisharkApiSFPPortsImplementation FibreMedia_M5;										/**< Fibre channel multi-mode 50um media support */
	ProfisharkApiSFPPortsImplementation FibreMedia_SM;										/**< Fibre channel single mode media support */
	ProfisharkApiSFPPortsImplementation FibreSpeed_1600;									/**< Fibre channel 1600MBytes/sec support */
	ProfisharkApiSFPPortsImplementation FibreSpeed_1200;									/**< Fibre channel 1200MBytes/sec support */
	ProfisharkApiSFPPortsImplementation FibreSpeed_800;										/**< Fibre channel 800MBytes/sec support */
	ProfisharkApiSFPPortsImplementation FibreSpeed_400;										/**< Fibre channel 400MBytes/sec support */
	ProfisharkApiSFPPortsImplementation FibreSpeed_200;										/**< Fibre channel 200MBytes/sec support */
	ProfisharkApiSFPPortsImplementation FibreSpeed_100;										/**< Fibre channel 100MBytes/sec support */

	float Temperature;	/**< Current module temperature in Celcius */
	float VCC;			/**< Current voltage in V */
	float TxBias;		/**< Current TX bias in mA */
	float TXPower;		/**< Current TX power in mW */
	float RXPower;		/**< Current RX power in mW */
	float LowAlarmTemperature;	/**< Low alarm threshold for module temperature in Celcius */
	float LowAlarmVCC;			/**< Low alarm threshold for voltage in V */
	float LowAlarmTxBias;		/**< Low alarm threshold for TX bias in mA */
	float LowAlarmTXPower;		/**< Low alarm threshold for TX power in mW */
	float LowAlarmRXPower;		/**< Low alarm threshold for RX power in mW */
	float LowWarningTemperature;	/**< Low warning threshold for module temperature in Celcius */
	float LowWarningVCC;			/**< Low warning threshold for voltage in V */
	float LowWarningTxBias;		/**< Low warning threshold for TX bias in mA */
	float LowWarningTXPower;		/**< Low warning threshold for TX power in mW */
	float LowWarningRXPower;		/**< Low warning threshold for RX power in mW */
	float HighAlarmTemperature;	/**< Low high threshold for module temperature in Celcius */
	float HighAlarmVCC;			/**< Low high threshold for voltage in V */
	float HighAlarmTxBias;		/**< Low high threshold for TX bias in mA */
	float HighAlarmTXPower;		/**< Low high threshold for TX power in mW */
	float HighAlarmRXPower;		/**< Low high threshold for RX power in mW */
	float HighWarningTemperature;	/**< Low high threshold for module temperature in Celcius */
	float HighWarningVCC;			/**< Low high threshold for voltage in V */
	float HighWarningTxBias;		/**< Low high threshold for TX bias in mA */
	float HighWarningTXPower;		/**< Low high threshold for TX power in mW */
	float HighWarningRXPower;		/**< Low high threshold for RX power in mW */
	uint32_t DiagFlags;					/**< bitfield containing alarm/warning and gpios flags. Values are from #ProfisharkApiSFPPortsDiagFlags */
};

/**
* \ingroup info
* \brief Describe all SFP ports
*/
struct ProfisharkApiSFPPortsStatus
{
	bool valid;								/**< The rest of the structure is only valid if 'valid' is true. False means SFP ports are not available */
	int portsCount;							/**< Number of available SFP ports */
	ProfisharkApiSFPPortStatus ports[2];
};

/**
* \ingroup info
* \brief State of the master/slave resolution
*/
enum ProfisharkApiPHYRegistersMasterSlave
{
	ProfisharkApiPHYRegistersMasterSlave_Master,
	ProfisharkApiPHYRegistersMasterSlave_Slave,
	ProfisharkApiPHYRegistersMasterSlave_NA,
};

/**
* \ingroup info
* \brief A PHY registers
*/
struct ProfisharkApiPHYRegisters
{
	ProfisharkApiPHYRegistersMasterSlave MasterSlave;	/**< Master/Slave resolution.*/
	ProfisharkAPIFlagState LPAutoNegCapable;
	ProfisharkAPIFlagState LPNextPageCapable;
	ProfisharkAPIFlagState NextPageRequest;
	ProfisharkAPIFlagState Acknowledge;
	ProfisharkAPIFlagState Advertise1000BASET_FDX;
	ProfisharkAPIFlagState Advertise1000BASET_HDX;
	ProfisharkAPIFlagState Advertise100BASETX_FDX;
	ProfisharkAPIFlagState Advertise100BASETX_HDX;
	ProfisharkAPIFlagState Advertise10BASET_FDX;
	ProfisharkAPIFlagState Advertise10BASET_HDX;
	ProfisharkAPIFlagState AdvertiseAsymmetricPause;
	ProfisharkAPIFlagState AdvertiseSymmetricPause;
	ProfisharkAPIFlagState ParallelDetectionFault;
	ProfisharkAPIFlagState RemoteFault;
	ProfisharkAPIFlagState MasterSlaveFault;
	ProfisharkAPIFlagState LocalReceiver;
	ProfisharkAPIFlagState RemoteReceiver;
	uint8_t IdleErrorCount;
	ProfisharkAPIFlagState _100BASETX_LockError;
	ProfisharkAPIFlagState _100BASETX_ReceiveError;
	ProfisharkAPIFlagState _100BASETX_TransmitError;
	ProfisharkAPIFlagState _100BASETX_SSDError;
	ProfisharkAPIFlagState _100BASETX_ESDError;
	ProfisharkAPIFlagState _1000BASET_LockError;
	ProfisharkAPIFlagState _1000BASET_ReceiveError;
	ProfisharkAPIFlagState _1000BASET_TransmitError;
	ProfisharkAPIFlagState _1000BASET_SSDError;
	ProfisharkAPIFlagState _1000BASET_ESDError;
	ProfisharkAPIFlagState _1000BASET_CarrierExtensionError;
	ProfisharkAPIFlagState MDICrossoverError;
};

/**
* \ingroup info
* \brief Describe all available PHYs registers
*/
struct ProfisharkApiPHYsRegisters
{
	bool valid;										/**< The rest of the structure is only valid if 'valid' is true. False means PHYs registers are not available */
	int portsCount;									/**< Number of available PHYs */
	ProfisharkApiPHYRegisters phyRegisters[2];		/**< Array containing the PHYs registers */
};

/**
* \ingroup info
* \brief Device link status
*/
struct ProfisharkApiLinkStatus
{
	ProfisharkApiRJ45PortsStatus RJ45PortsStatus;	/**< Status of the RJ45 ports (if any) */
	ProfisharkApiSFPPortsStatus SFPPortsStatus;		/**< Status of the SFP ports (if any) */
	ProfisharkApiPHYsRegisters PHYRegisters;		/**< PHYs registers (if available) */
};

/**
* \ingroup statistics
* \brief Statistics model version 
*/
enum ProfisharkApiStatisticsVersion
{
	ProfisharkApiStatisticsVersion_1=1,
	ProfisharkApiStatisticsVersion_2=2,
};

/**
* \ingroup statistics
* \brief Statistics structure for a single port of a device supporting V1 model
*/
struct ProfisharkApiStatisticsV1Port
{
	uint64_t LessThan64;		/**< Number of packets of size < 64 */
	uint64_t Betwen64And1518;	/**< Number of packets of size between 64 and 1518 */
	uint64_t MoreThan1518;		/**< Number of packets of size gretaer than 1518 */
	uint64_t Collision;			/**< Number of collisions detected */
	uint64_t CrcError;			/**< Number of packets containing a CRC error */
	uint64_t Jabber;			/**< Number of packets of size < 64 containing a CRC error */
	uint64_t Bytes;				/**< Number of bytes transferred */
	uint64_t MaximumBytes;		/**< Maximum number of bytes that could have passed through the interface at current link speed since last time statistics where retrieved. Can be used to determine bandwidth usage. */
	uint64_t TotalValid;		/**< Number of valid packets */
	uint64_t TotalInvalid;		/**< Number of invalid packets */
	uint64_t HardwareDropped;	/**< Number of packets dropped by the hardware */
};

/**
* \ingroup statistics
* \brief Statistics structure for devices supporting V1 model
*/
struct ProfisharkApiStatisticsV1
{
	uint64_t Date;										/**< Date when the statistics was gathered in ms since the system started */
	uint64_t SoftwareDropped;							/**< Number of packets dropped by the driver */
	uint64_t LinkUp;									/**< Duration the link has been up on the port in ms */
	uint64_t LinkDown;									/**< Duration the link has been down last time it happened in ms */
	ProfisharkApiStatisticsV1Port PortsStatistics[2];	/**< Statistics for each port */
	uint64_t BogusBytes;								/**< Bogus bytes received by the driver */
	uint64_t TotalPackets;								/**< Total number of packets parsed by the driver */
};

/**
* \ingroup statistics
* \brief The maximum number of counters available in a device supporting V2 statistics model.
*/
#define PROFISHARKAPI_STATISTICSV2_COUNTERS 16

/**
* \ingroup memorycapture
* \brief Minimum length of a buffer for memory capture
*/
#define PROFISHARKAPI_MEMORYCAPTURE_BUFFER_LENGTH (16384*4*8)

/**
* \ingroup statistics
* \brief Statistics structure for devices supporting V2 model
*/
struct ProfisharkApiStatisticsV2
{
	uint64_t Date;												/**< Date when the statistics was gathered in ms since the system started */
	uint64_t SoftwareDropped;									/**< Number of packets dropped by the driver */
	uint64_t Counters[PROFISHARKAPI_STATISTICSV2_COUNTERS];		/**< The counters values */
	uint64_t BogusBytes;								/**< Bogus bytes received by the driver */
	uint64_t TotalPackets;								/**< Total number of packets parsed by the driver */
};

/**
* \ingroup statistics
* \brief Statistics structure for a Profishark device
*/
struct ProfisharkApiStatistics
{
	ProfisharkApiStatisticsVersion version; /**< Version of the structure. Defines which member of the union is used */
	/** 
	* \brief This union contains the statistics. 
	*
	* Depending on the value of version a different member of the union is filled.
	* If version == ProfisharkApiStatisticsVersion_1, StatisticsV1 is filled.
	* If version == ProfisharkApiStatisticsVersion_2, StatisticsV2 is filled.
	*/
	union
	{
		ProfisharkApiStatisticsV1 StatisticsV1;	/**< Statistics for devices supporting V1 model */
		ProfisharkApiStatisticsV2 StatisticsV2; /**< Statistics for devices supporting V2 model */
	};
};

/**
* \ingroup statistics
* \brief Counter V2 size comparison mode
*/
enum ProfisharkApiCounterV2SizeMode
{
	ProfisharkApiCounterV2SizeMode_SizeModeDontCare=-1,		/**< Counter does not care about packet size */
	ProfisharkApiCounterV2SizeMode_SizeModeGreaterThan,		/**< Counter counts packet if packet size > 'size' */
	ProfisharkApiCounterV2SizeMode_SizeModeLessThan,		/**< Counter counts packet if packet size < 'size' */
	ProfisharkApiCounterV2SizeMode_SizeModeEqual,			/**< Counter counts packet if packet size = 'size' */
};


/**
* \ingroup statistics
* \brief Counter V2 mode
*/
enum ProfisharkApiCounterV2Mode
{
	ProfisharkApiCounterV2Mode_CounterModePackets,	/**< Counter is counting packets */
	ProfisharkApiCounterV2Mode_CounterModeBytes,	/**< Counter is counting bytes */
};

/**
* \ingroup statistics
* \brief Counter V2 description
*
* For every ProfisharkAPIState member, ProfisharkAPIState_Enabled means a packet is counted if the condition is true,
* ProfisharkAPIState_Disabled means a packet is counted if the condition is false
* and ProfisharkAPIState_DontCare means it is counted no matter what.
*/
struct ProfisharkApiCounterV2
{
	int n;										/**< Counter id */
	ProfisharkAPIState isIPv4;					/**< Test if a packet is IPv4 */
	ProfisharkAPIState isIPv6;					/**< Test if a packet is IPv6 */
	ProfisharkAPIState isbcast;					/**< Test if a packet is broadcast */
	ProfisharkAPIState ismcast;					/**< Test if a packet is multicast */
	ProfisharkAPIState isucast;					/**< Test if a packet is unicast */
	ProfisharkAPIState isicmp;					/**< Test if a packet is ICMP */
	ProfisharkAPIState isudp;					/**< Test if a packet is UDP */
	ProfisharkAPIState istcp;					/**< Test if a packet is TCP */
	ProfisharkAPIState iscrcValid;				/**< Test if a packet has valid CRC */
	ProfisharkAPIState isDropped;				/**< Test if a packet was dropped by hardware */
	int size;									/**< The size to compare packet size with */
	ProfisharkApiCounterV2SizeMode sizeMode;	/**< Comparison operation between 'size' and packet size */
	uint8_t ip[16];								/**< Compare 'ip' with the packet IP address. LSB is ip[0]. For IPv4, use ip[0..3] and set ipMask accordingly. */
	uint8_t ipMask[16];							/**< Mask for the Ip comparision */
	ProfisharkAPIState port1;					/**< Counter enabled on port A */
	ProfisharkAPIState port2;					/**< Counter enabled on port B */
	ProfisharkApiCounterV2Mode mode;			/**< Counter mode (packets or bytes) */
};

/**
* \ingroup filters
* \brief DPI filter configuration
*/
struct ProfisharkApiDpiFilter
{
	ProfisharkAPIState Enabled;		/**< State of the DPI filter */
	uint8_t Data[32];				/**< Data to search in packet. Only the first 16 bytes are currently used */
	uint32_t Mask;					/**< Mask for the data field. Each bit correspond to a byte of Data (i.e. bit n of Mask defines if Data[n] is enabled) */
};

/**
* \ingroup filters
* \brief Flags that enable of disable filters fields
*/
enum ProfisharkApiPacketFilterFlags
{
	ProfisharkApiPacketFilterFlags_isIPV6 = 0x1,			/**< Source and destination IP addresses are in IPv6 format */
	ProfisharkApiPacketFilterFlags_dstMacEnabled = 0x2,		/**< Destination mac filter is enabled */
	ProfisharkApiPacketFilterFlags_srcMacEnabled = 0x4,		/**< Source mac filter is enabled */
	ProfisharkApiPacketFilterFlags_dstIpEnabled = 0x8,		/**< Destination IP filter is enabled */
	ProfisharkApiPacketFilterFlags_srcIpEnabled = 0x10,		/**< Source IP filter is enabled */
	ProfisharkApiPacketFilterFlags_dstPortEnabled = 0x20,	/**< Destination port filter is enabled */
	ProfisharkApiPacketFilterFlags_srcPortEnabled = 0x40,	/**< Source port filter is enabled */
};

/**
* \ingroup filters
* \brief Modes defining the comportement of packet filter fields
*/
enum ProfisharkApiPacketFilterMode
{
	ProfisharkApiPacketFilterMode_macIsOr = 0x1,		/**< Only DestinationMAC is used and is compared to both the packet source and destination MAC address */
	ProfisharkApiPacketFilterMode_ipIsOr = 0x2,			/**< Only DestinationIP is used and is compared to both the packet source and destination IP address */
	ProfisharkApiPacketFilterMode_portIsOr = 0x4,		/**< Only DestinationPort is used and is compared to both the packet source and destination port */
};

/**
* \ingroup filters
* \brief Packet filter configuration
*/
struct ProfisharkApiPacketFilter
{
	ProfisharkAPIState FilterEnabled;	/**< Enable or disable the filter */
	uint32_t flags;						/**< bit field containing #ProfisharkApiPacketFilterFlags values */
	uint16_t mode;						/**< bit field containing #ProfisharkApiPacketFilterMode values */
	uint8_t DestinationMAC[6];			/**< Destination MAC, DestinationMAC[0] is LSB. */
	uint8_t SourceMAC[6];				/**< Source MAC, SourceMAC[0] is LSB. */
	uint8_t DestinationIP[16];			/**< Destination IP, DestinationIP[0] is LSB. If IPv4, use only bytes 0..3.*/
	uint8_t SourceIP[16];				/**< Source IP, SourceIP[0] is LSB. If IPv4, use only bytes 0..3.*/
	uint16_t DestinationPort;			/**< Destination Port */
	uint16_t SourcePort;				/**< Source Port */
	unsigned int DestinationMACMask:6;			/**< Mask for destination MAC. Each bit correspond to a byte of DestinationMAC */
	unsigned int SourceMACMask:6;				/**< Mask for Source MAC . Each bit correspond to a byte of SourceMAC */
	unsigned int DestinationIPMask:16;			/**< Mask for destination IP. Each bit correspond to a byte of DestinationIP */
	unsigned int SourceIPMask:16;				/**< Mask for Source IP . Each bit correspond to a byte of SourceIP */
	unsigned int DestinationPortMask:2;			/**< Mask for destination Port. Each bit correspond to a byte of DestinationPort */
	unsigned int SourcePortMask:2;				/**< Mask for Source Port . Each bit correspond to a byte of SourcePort */
};

/**
* \ingroup timing
* \brief Timing configuration
*/
struct ProfisharkAPITimingTime
{
	/**
	* \brief Time Source
	* 
	* Can only be ProfisharkAPITimeSource_Manual, ProfisharkAPITimeSource_SNTP or ProfisharkAPITimeSource_GPS
	* ProfisharkAPITimeSource_Manual and ProfisharkAPITimeSource_SNTP work similarly, 
	* but time will be reported as coming from either a manual source or from SNTP when retrieving timing info. 
	* sec and nsec must be filled with seconds since 01/01/1970 and nanoseconds since the beginning of that second.
	* ProfisharkAPITimeSource_GPS does not need sec and nsec to be filled. It instructs the board to load the time from GPS ass soon as possible.
	*/
	ProfisharkAPITimeSource source;
	uint32_t sec;		/**< Seconds since 01/01/1970 */
	uint32_t nsec;		/**< Nanoseconds */
};

/**
* \ingroup timing
* \brief State of device timing board
*/
enum ProfisharkAPITimingBoardState
{
	ProfisharkAPITimingBoardState_Available,	/**< Timing board connected and available */
	ProfisharkAPITimingBoardState_Absent,		/**< Timing board not connected */
	ProfisharkAPITimingBoardState_Locked,		/**< Timing board connected but not available */
};

/**
* \ingroup timing
* \brief State of the gps chip
*/
enum ProfisharkAPITimingFixState
{
	ProfisharkAPITimingFixState_NoFix,	/**< No GPS fix */
	ProfisharkAPITimingFixState_2DFix,	/**< GPS 2D fix */
	ProfisharkAPITimingFixState_3DFix,	/**< GPS 3D fix */
};

/**
* \ingroup timing
* \brief Represents a date and time
*/
struct ProfisharkAPITimingDataTime
{
	int hour;
	int minute;
	int second;
	int year;
	int month;
	int day;
};

/**
* \ingroup timing
* \brief Status of the device timing board
*/
struct ProfisharkAPITimingStatus
{
	ProfisharkAPITimingBoardState TimingBoardState;	/**< Timing board state. The rest of the structure is valid only if this is ProfisharkAPITimingBoardState_Available */
	ProfisharkAPITimingFixState Fix;				/**< GPS fix state */
	int UsedSats;									/**< Number of used satellites */
	int GPSSats;									/**< Total GPS satellites in sight */
	int GLONASSSats;								/**< Total GLONASS satellites in sight */
	ProfisharkAPIFlagState GPSPPS;			/**< State of the GPS PPS signal */
	ProfisharkAPIFlagState ExternalPPS;		/**< State of the external PPS signal */
	ProfisharkAPIFlagState TimestampSync;		/**< State of the timestamp synchronization */
	double Deviation;								/**< Deviation of timestamp from PPS in ns */
	int Precision;									/**< Estimated GPS PPS precision in ns */
	ProfisharkAPITimeSource TimeSource;				/**< Time information source */
	struct ProfisharkAPITimingDataTime CurrentGPSTime; /**< Current time according to the module GPS */
};

/**
* \ingroup diskcapture
* \ingroup memorycapture
* \brief Output file format
*/
enum ProfisharkAPINetworkDataFormat
{
	ProfisharkAPINetworkDataFormat_PcapNG,		/**< Pcap-ng */
};

/**
* \ingroup diskcapture
*\ brief Direct capture configuration
*/
struct ProfisharkAPIDiskDumpConfig
{
	ProfisharkAPIState Loop;						/**< Enable round-robin style loop through files, erasing older files if needed.*/
	ProfisharkAPIState StopIfCacheFull;				/**< Stop capture automatically when cache is full */
	ProfisharkAPINetworkDataFormat FileFormat;		/**< Capture file format. Not all formats are available for each device. Check in devices capabilities. */
	wchar_t * FileName;							/**< Output file name. Timestamp is automatically added to the file name before the file extension. */
	uint64_t MaximumFileSize;						/**< Maximum size of a capture file in bytes. 0 means no limit */
	uint64_t MaximumCacheSize;						/**< Maximum cache size in bytes */
	int FilesNumber;								/**< Maximum number of files generated by the capture. 0 means no limit */
	uint32_t MaximumFileDuration;					/**< Maximum duration of a capture file in seconds */
};

/**
* \ingroup diskcapture
* \brief Direct capture status
*/
struct ProfisharkAPIDiskDumpStatus
{
	bool CaptureIsRunning;			/**< true if the capture is currently running, false otherwise */
	bool HighCacheUsage;			/**< true if cache is heavily used, false otherwise */
	uint64_t UsedCache;				/**< Number of bytes used in the cache */
	uint64_t BytesWritten;			/**< Total bytes written to disk */
	uint64_t BytesDropped;			/**< Total bytes dropped by software */
	ProfisharkAPIRet ErrorStatus;	/**< Last direct capture error */
	uint64_t MaximumCacheSize;		/**< Maximum cache size in bytes */
	uint64_t FilesWritten;			/**< Number of files fully written (excluding the file being currently written to) */
};

/**
* \ingroup info
* \brief Maximum length of a HW firmware name.
*/
#define PROFISHARK_MAX_HW_NAME_LENGTH 32

/**
* \ingroup info
* \brief Maximum number of HW firmwares.
*/
#define PROFISHARK_MAX_HW_COUNT 4

/**
* \ingroup info
* \brief Descriptions of a HW firmwares
*/
struct ProfisharkAPIDeviceHWFirmwareDesc
{
	char name[PROFISHARK_MAX_HW_NAME_LENGTH];	/**< Name of the firmware */
};

/**
* \ingroup info
* \brief Descriptions of the supported HW firmwares
*/
struct ProfisharkAPIDeviceHWFirmwaresDesc
{
	int count;						/**< Number of HW firmwares supported by the device. Only the count first firmware descriptions are valid in this structure */
	ProfisharkAPIDeviceHWFirmwareDesc Firmwares[PROFISHARK_MAX_HW_COUNT];	/**< Description of each HW firmware */
};

/**
* \ingroup callbacks
*\ brief Global callback types
*/
enum ProfisharkApiGlobalCallbackType
{
	ProfisharkApiGlobalCallbackType_NewDevice,		/**< Function is called on device insertion. ProfisharkApiCallbackValue is not used */
	ProfisharkApiGlobalCallbackType_DeviceRemoved,	/**< Function is called on device removal. ProfisharkApiCallbackValue is not used */
};

/**
* \ingroup callbacks
*\ brief Device specific callback types
*/
enum ProfisharkApiDeviceCallbackType
{
	ProfisharkApiDeviceCallbackType_DiskDump,	/**< Function is called on direct capture events (capture start, capture stop). ProfisharkApiCallbackValue is to be cast as \ref ProfisharkApiCallbackValue_DiskDump */
};

/**
* \ingroup memorycapture
*\ brief Memory capture configuration
*/
struct ProfisharkAPIMemoryCaptureConfig
{
	ProfisharkAPINetworkDataFormat Format;		/**< Capture format. Not all formats are available for each device. Check in devices capabilities. */
};

/**
* \brief Returns if the value means success
* \param a ProfisharkAPIRet value
* \return whether the ProfisharkAPIRet corresponds to a success
*/
#define IS_PROFISHARK_API_SUCCESS(a) (a==ProfisharkAPIRet_Success)

/**
* \ingroup detect
* \brief Profishark device handle.
*
* Opaque type used in most of the API calls.
*/
typedef void * ProfisharkApiDeviceHandle;

/**
* \brief Invalid handle value
*/
#define PROFISHARK_API_INVALID_HANDLE ((void*)-1)

/**
* \ingroup callbacks
* \brief Profishark callback value.
*
* Opaque type used by callback functions.
* To be used, this needs to be cast to the proper type depending on the callback type
*/
typedef void * ProfisharkApiCallbackValue;

/**
* \ingroup callbacks
* \brief Callback function
* \param handle a Profishark device handle
* \param context a user defined value
* \deprecated Functions using this type are deprecated. See \ref ProfisharkApiCallbackExt.
*/
typedef void (*ProfisharkApiCallback)(ProfisharkApiDeviceHandle, void*);

/**
* \ingroup callbacks
* \brief Callback function
* \param handle a Profishark device handle
* \param value a value passed to the callback. To use it, the user has to cast it to the appropriate type depending on the callback type. See \ref ProfisharkApiGlobalCallbackType and \ref ProfisharkApiDeviceCallbackType
* \param context a user defined value
*/
typedef void (*ProfisharkApiCallbackExt)(ProfisharkApiDeviceHandle, ProfisharkApiCallbackValue *, void*);

/**
* \ingroup misc
* \brief Sleeps for ms milliseconds.
*/
PROFISHARKAPI_API void ProfisharkApiSleepMs(int ms);

/**
* \ingroup management
* \brief Get API version.
*/
PROFISHARKAPI_API ProfisharkApiSWVersion ProfisharkAPIGetVersion(void);

/**
* \ingroup management
* \brief Initialize the Profishark library.
*
* This must be called before any other API call can be made.
* Only #ProfisharkAPIGetVersion, #ProfisharkAPIAddGlobalCallback, #ProfisharkAPIAddDeviceCallback, #ProfisharkAPIRemoveGlobalCallback and #ProfisharkAPIRemoveDeviceCallback can be called before #ProfisharkAPIInitialize.
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIInitialize(void);

/**
* \ingroup management
* \brief Cleanup internal library structures. 
* 
* No API calls can be made after this except for Initialize().
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIUninitialize(void);

struct ProfisharkApiDeviceNode
{
	struct ProfisharkApiDeviceNode * next;
	ProfisharkApiDeviceHandle handle;
};

struct ProfisharkApiDeviceList
{
	struct ProfisharkApiDeviceNode * dev;
	int count;
};

PROFISHARKAPI_API void ProfisharkApiDeviceListFree(ProfisharkApiDeviceList * list);

/**
* \ingroup detect
* \brief Get a list containing the currently connected Profishark devices
* \return A list of the handles for currently connected Profishark devices
*/
PROFISHARKAPI_API ProfisharkApiDeviceList * ProfisharkAPIGetDevicesList(void);

/**
* \ingroup callbacks
* \brief Add a new function to be called when a new Profishark device is detected
* \param callback the function to be called
* \param context the parameter passed to the function to be called
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_Exist if the function is already registered
* \deprecated Deprecated, use ProfisharkAPIAddGlobalCallback
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIAddNewDeviceCallback(ProfisharkApiCallback callback, void * context);

/**
* \ingroup callbacks
* \brief Remove a function from the list of functions called when a new Profishark device is detected
* \param callback the function to be removed
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NotExist if the function was not registered
* \deprecated Deprecated, use ProfisharkAPIRemoveGlobalCallback
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIRemoveNewDeviceCallback(ProfisharkApiCallback callback);

/**
* \ingroup callbacks
* \brief Add a new function to be called when a Profishark device is disconnected
* \param callback the function to be called
* \param context the parameter passed to the function to be called
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_Exist if the function is already registered
* \deprecated Deprecated, use ProfisharkAPIAddGlobalCallback
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIAddDeviceGoneCallback(ProfisharkApiCallback callback, void * context);

/**
* \ingroup callbacks
* \brief Remove a function from the list of functions called when a new Profishark device is disconnected
* \param callback the function to be removed
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NotExist if the function was not registered
* \deprecated Deprecated, use ProfisharkAPIRemoveGlobalCallback
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIRemoveDeviceGoneCallback(ProfisharkApiCallback callback);

/**
* \ingroup info
* \brief Get device type
* \param handle a handle to the device
* \param[out] type a #ProfisharkAPIDeviceType that will be filled with the device type
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetDeviceType(ProfisharkApiDeviceHandle handle, ProfisharkAPIDeviceType & type);

/**
* \ingroup info
* \brief Get device MAC address
* \param handle a handle to the device
* \param[out] mac a 6 bytes array that will be filled with the device mac address
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetDeviceMAC(ProfisharkApiDeviceHandle handle, uint8_t * mac);

/**
* \ingroup info
* \brief Get device capabilities
* \param devType the type of the device
* \return a 64 bits integer that will hold a bitfield descibing the device capabilities. That bitfield values are described by #ProfisharkApiDeviceOptions.
*/
PROFISHARKAPI_API ProfisharkApiDeviceOptions ProfisharkAPIGetDeviceCapabilities(ProfisharkAPIDeviceType devType);

/**
* \ingroup info
* \brief Get device extra information
* \param handle a handle to the device
* \param[out] info a #ProfisharkAPIDeviceExtraInfo that will be filled with the device extra information
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetDeviceExtraInfo(ProfisharkApiDeviceHandle handle, ProfisharkAPIDeviceExtraInfo & info);

/**
* \ingroup features
* \brief Get device features
* \param handle a handle to the device
* \param[out] features a #ProfisharkAPIDeviceFeatures that will be filled by the information from the driver
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetDeviceFeatures(ProfisharkApiDeviceHandle handle, ProfisharkAPIDeviceFeatures & features);

/**
* \ingroup features
* \brief Set device features
* \param handle a handle to the device
* \param features a #ProfisharkAPIDeviceFeatures containing features to set
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
* \retval ProfisharkAPIRet_InvalidFirmware if the selected HW firmware id is not valid
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPISetDeviceFeatures(ProfisharkApiDeviceHandle handle, ProfisharkAPIDeviceFeatures features);

/**
* \ingroup features
* \brief Get the name corresponding to a HW firmware id
* \param handle a handle to the device
* \param desc a ProfisharkAPIDeviceHWFirmwareDesc that will hold firmware descriptions
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetHwFirmwaresDesc(ProfisharkApiDeviceHandle handle, ProfisharkAPIDeviceHWFirmwaresDesc & desc);

/**
* \ingroup features
* \brief Set network ports supported speeds
* \param handle a handle to the device
* \param speed a #ProfisharkAPIDevicePortsSpeed describing the supported speeds
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPISetDevicePortsSpeed(ProfisharkApiDeviceHandle handle, ProfisharkAPIDevicePortsSpeed speed);

/**
* \ingroup features
* \brief Get network ports supported speeds
* \param handle a handle to the device
* \param speed a #ProfisharkAPIDevicePortsSpeed that will be filled with supported speeds
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetDevicePortsSpeed(ProfisharkApiDeviceHandle handle, ProfisharkAPIDevicePortsSpeed & speed);

/**
* \ingroup features
* \brief Save device features and ports speed to EEPROM
* \param handle a handle to the device
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPISaveDeviceFeatures(ProfisharkApiDeviceHandle handle);

/**
* \ingroup features
* \brief Restart the device
* \param handle a handle to the device
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIRestartDevice(ProfisharkApiDeviceHandle handle);

/**
* \ingroup info
* \brief Get device firmwares and driver versions
* \param handle a handle to the device
* \param[out] versions a #ProfisharkApiVersions that will hold versions information
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetVersions(ProfisharkApiDeviceHandle handle, ProfisharkApiVersions & versions);

/**
* \ingroup info
* \brief Get device link status
* \param handle a handle to the device
* \param[out] status a #ProfisharkApiLinkStatus that will hold link information
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetLinkStatus(ProfisharkApiDeviceHandle handle, ProfisharkApiLinkStatus & status);

/**
* \ingroup statistics
* \brief Get device statistics
* \param handle a handle to the device
* \param[out] statistics a #ProfisharkApiStatistics that will hold statistics
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetStatistics(ProfisharkApiDeviceHandle handle, ProfisharkApiStatistics & statistics);

/**
* \ingroup statistics
* \brief Set device counter configuration
* \param handle a handle to the device
* \param counter the #ProfisharkApiCounterV2 structure containing the counter configuration
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPISetCounterV2(ProfisharkApiDeviceHandle handle, ProfisharkApiCounterV2 counter);

/**
* \ingroup statistics
* \brief Get device counter configuration
* \param handle a handle to the device
* \param n the id of the counter to retrieve
* \param[out] counter the #ProfisharkApiCounterV2 structure that will contain the counter configuration
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetCounterV2(ProfisharkApiDeviceHandle handle, int n, ProfisharkApiCounterV2 & counter);

/**
* \ingroup filters
* \brief Set DPI filter configuration
* \param handle a handle to the device
* \param filter a #ProfisharkApiDpiFilter holding dpi filter configuration
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPISetDpiFilter(ProfisharkApiDeviceHandle handle, ProfisharkApiDpiFilter filter);

/**
* \ingroup filters
* \brief Get DPI filter configuration
* \param handle a handle to the device
* \param[out] filter a #ProfisharkApiDpiFilter that will hold the dpi filter configuration
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetDpiFilter(ProfisharkApiDeviceHandle handle, ProfisharkApiDpiFilter & filter);

/**
* \ingroup filters
* \brief Save DPI filter configuration
* \param handle a handle to the device
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPISaveDpiFilter(ProfisharkApiDeviceHandle handle);

/**
* \ingroup filters
* \brief Set packet filter configuration
* \param handle a handle to the device
* \param filter a #ProfisharkApiPacketFilter holding packet filter configuration
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPISetPacketFilter(ProfisharkApiDeviceHandle handle, ProfisharkApiPacketFilter filter);

/**
* \ingroup filters
* \brief Get packet filter configuration
* \param handle a handle to the device
* \param[out] filter a #ProfisharkApiPacketFilter that will hold the packet filter configuration
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetPacketFilter(ProfisharkApiDeviceHandle handle, ProfisharkApiPacketFilter & filter);

/**
* \ingroup filters
* \brief Save packet filter configuration
* \param handle a handle to the device
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPISavePacketFilter(ProfisharkApiDeviceHandle handle);

/**
* \ingroup timing
* \brief Set timestamp time information
* \param handle a handle to the device
* \param time a #ProfisharkAPITimingTime that contains the time information
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPISetTimingTime(ProfisharkApiDeviceHandle handle, ProfisharkAPITimingTime time);

/**
* \ingroup timing
* \brief Get timing board status
* \param handle a handle to the device
* \param[out] status a #ProfisharkAPITimingStatus that will contain the timing board status
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGetTimingStatus(ProfisharkApiDeviceHandle handle, ProfisharkAPITimingStatus & status);

/**
* \ingroup diskcapture
* \brief Set direct capture configuration
* \param handle a handle to the device
* \param config a #ProfisharkAPIDiskDumpConfig that contains the direct capture configuration
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIDiskDumpSetConfig(ProfisharkApiDeviceHandle handle, ProfisharkAPIDiskDumpConfig config);

/**
* \ingroup diskcapture
* \brief Start direct capture
* \param handle a handle to the device
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIDiskDumpStart(ProfisharkApiDeviceHandle handle);

/**
* \ingroup diskcapture
* \brief Stop direct capture
* \param handle a handle to the device
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIDiskDumpStop(ProfisharkApiDeviceHandle handle);

/**
* \ingroup diskcapture
* \brief Get direct capture status
* \param handle a handle to the device
* \param[out] status a #ProfisharkAPIDiskDumpStatus that will contain the direct capture status
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIDiskDumpGetStatus(ProfisharkApiDeviceHandle handle, ProfisharkAPIDiskDumpStatus &status);

/**
* \ingroup memorycapture
* \brief Start memory capture
* \param handle a handle to the device
* \param config a #ProfisharkAPIMemoryCaptureConfig that contains the memory capture configuration
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIMemoryCaptureStart(ProfisharkApiDeviceHandle handle, ProfisharkAPIMemoryCaptureConfig config);

/**
* \ingroup memorycapture
* \brief Stop memory capture
* \param handle a handle to the device
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIMemoryCaptureStop(ProfisharkApiDeviceHandle handle);

/**
* \ingroup memorycapture
* \brief Get network data
* \param handle a handle to the device
* \param[out] buffer a buffer to store the data in the requested format. It must be at least #PROFISHARKAPI_MEMORYCAPTURE_BUFFER_LENGTH long.
* \param[in,out] length the length of the buffer. It will be filled with the length of the returned data.
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
* \retval ProfisharkAPIRet_TooSmall if the buffer is too small
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIMemoryCaptureGet(ProfisharkApiDeviceHandle handle, uint8_t * buffer, int & length);

/**
* \ingroup memorycapture
* \brief Get the file header corresponding to the requested format
* \param handle a handle to the device
* \param[out] buffer a buffer to store the header. It must be at least #PROFISHARKAPI_MEMORYCAPTURE_BUFFER_LENGTH long.
* \param[in,out] length the length of the buffer. It will be filled with the length of the returned data.
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
* \retval ProfisharkAPIRet_TooSmall if the buffer is too small
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIMemoryCaptureGetHeader(ProfisharkApiDeviceHandle handle, uint8_t * buffer, int & length);

/**
* \ingroup timing
* \brief Flush GPS UART output interface
* \param handle a handle to the device
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
* \retval ProfisharkAPIRet_Unsupported if the device does not support this operation
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGPSOutputFlush(ProfisharkApiDeviceHandle handle);

/**
* \ingroup timing
* \brief Fetch data from the GPS UART output interface
* \param handle a handle to the device
* \param data a buffer to be filled with data from the gps uart interface
* \param the size of the buffer
* \param[out] the number of bytes transferred from the interface
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NoDev if the device does not exist
* \retval ProfisharkAPIRet_Unsupported if the device does not support this operation
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIGPSOutputRead(ProfisharkApiDeviceHandle handle, char * data, int len, int * transferred);

/**
* \ingroup callbacks
* \brief Add a global callback function.
* \param type the type of callback to add. This defines when the function is called.
* \param callback the function to be called
* \param context a pointer passed to the callback function
* \retval ProfisharkAPIRet_Success on success
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIAddGlobalCallback(ProfisharkApiGlobalCallbackType type, ProfisharkApiCallbackExt callback, void * context);

/**
* \ingroup callbacks
* \brief Add a device callback function.
* \param handle a handle to the device
* \param type the type of callback to add. This defines when the function is called.
* \param callback the function to be called
* \param context a pointer passed to the callback function
* \retval ProfisharkAPIRet_Success on success
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIAddDeviceCallback(ProfisharkApiDeviceHandle handle, ProfisharkApiDeviceCallbackType type, ProfisharkApiCallbackExt callback, void * context);

/**
* \ingroup callbacks
* \brief Remove a global callback function.
* \param type the type of callback to remove. Must be the same as the one used for the #ProfisharkAPIAddGlobalCallback call.
* \param callback the function to be removed
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NotExist if the function was not registered
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIRemoveGlobalCallback(ProfisharkApiGlobalCallbackType type, ProfisharkApiCallbackExt callback);

/**
* \ingroup callbacks
* \brief Remove a device callback function.
* \param handle a handle to the device
* \param type the type of callback to remove. Must be the same as the one used for the #ProfisharkAPIAddDeviceCallback call.
* \param callback the function to be removed
* \retval ProfisharkAPIRet_Success on success
* \retval ProfisharkAPIRet_NotExist if the function was not registered
*/
PROFISHARKAPI_API ProfisharkAPIRet ProfisharkAPIRemoveDeviceCallback(ProfisharkApiDeviceHandle handle, ProfisharkApiDeviceCallbackType type, ProfisharkApiCallbackExt callback);

/**
* \ingroup callbacks
*\ brief A structure describing the event passed to a ProfisharkApiDeviceCallbackType_DiskDump callback.
*/
struct ProfisharkApiCallbackValue_DiskDump
{
	bool CaptureIsRunning;	/**< true if capture is running. False otherwise. */
	ProfisharkAPIRet ErrorStatus;	/**< Contains the error code of the last direct capture error. */
};

}
